#!/bin/bash

# سیستم حسابداری دزکالا - اسکریپت نصب خودکار
# DezKala Accounting System - Automated Installation Script

set -e

echo "=========================================="
echo "سیستم حسابداری دزکالا"
echo "Automated Installation Script"
echo "=========================================="

# رنگ‌های output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# متغیرها
INSTALLATION_DIR="/home/dezkalac/public_html/acc"
LOG_FILE="$INSTALLATION_DIR/install.log"

# توابع
log() {
    echo -e "${GREEN}[✓]${NC} $1" | tee -a "$LOG_FILE"
}

error() {
    echo -e "${RED}[✗]${NC} $1" | tee -a "$LOG_FILE"
}

warning() {
    echo -e "${YELLOW}[!]${NC} $1" | tee -a "$LOG_FILE"
}

# شروع
echo "$(date)" > "$LOG_FILE"
log "شروع نصب سیستم حسابداری"

# بررسی PHP
if ! command -v php &> /dev/null; then
    error "PHP نصب نشده است"
    exit 1
fi
PHP_VERSION=$(php -v | head -n 1 | awk '{print $2}')
log "PHP نسخه $PHP_VERSION یافت شد"

# بررسی MySQL
if ! command -v mysql &> /dev/null; then
    error "MySQL نصب نشده است"
    exit 1
fi
log "MySQL یافت شد"

# بررسی Composer
if ! command -v composer &> /dev/null; then
    error "Composer نصب نشده است"
    exit 1
fi
log "Composer یافت شد"

# قدم 1: کپی فایل .env
log "تنظیم فایل .env..."
if [ ! -f "$INSTALLATION_DIR/.env" ]; then
    cp "$INSTALLATION_DIR/.env.example" "$INSTALLATION_DIR/.env"
    log ".env ایجاد شد"
else
    warning ".env قبلاً وجود دارد"
fi

# قدم 2: تولید کلید‌ها
log "تولید کلیدهای اپلیکیشن..."
cd "$INSTALLATION_DIR"
php artisan key:generate --force 2>&1 | tee -a "$LOG_FILE"
php artisan jwt:secret --force 2>&1 | tee -a "$LOG_FILE"
log "کلیدها تولید شدند"

# قدم 3: نصب Composer Dependencies
log "نصب Composer dependencies..."
composer install --no-dev --optimize-autoloader 2>&1 | tee -a "$LOG_FILE"
log "Dependencies نصب شدند"

# قدم 4: ایجاد دیتابیس (اختیاری - اگر فایل .my.cnf موجود باشد)
if [ -f ~/.my.cnf ]; then
    log "سعی در ایجاد دیتابیس..."
    mysql -e "CREATE DATABASE IF NOT EXISTS dezkalac_accounting CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;" 2>&1 | tee -a "$LOG_FILE" || warning "دیتابیس قبلاً وجود دارد یا خطای دسترسی"
    log "دیتابیس بررسی شد"
else
    warning "فایل .my.cnf یافت نشد - لطفاً دیتابیس را به دستی ایجاد کنید"
fi

# قدم 5: اجرا Migrations
log "اجرای migrations..."
php artisan migrate --force 2>&1 | tee -a "$LOG_FILE" || error "خطا در migrations"
log "Migrations اجرا شدند"

# قدم 6: تنظیم مجوزهای فایل‌ها
log "تنظیم مجوزهای فایل‌ها..."
chmod -R 755 "$INSTALLATION_DIR"
chmod -R 777 "$INSTALLATION_DIR/storage"
chmod -R 777 "$INSTALLATION_DIR/bootstrap/cache"
log "مجوزها تنظیم شدند"

# قدم 7: Cache optimization
log "Optimization..."
php artisan config:cache 2>&1 | tee -a "$LOG_FILE"
php artisan view:cache 2>&1 | tee -a "$LOG_FILE"
log "Optimization کامل شد"

# قدم 8: ایجاد Admin User
log "ایجاد Admin User..."
php artisan tinker << 'EOF'
$user = new App\Models\User();
$user->name = 'Admin';
$user->email = 'admin@dezkala.com';
$user->phone = '09166809066';
$user->role = 'admin';
$user->password = Hash::make('admin@12345');
$user->warehouse_id = 1;
$user->status = 'active';
$user->save();
echo "Admin user created successfully\n";
exit();
EOF
log "Admin user ایجاد شد"

# پایان
echo ""
log "=========================================="
log "✓ نصب موفق‌آمیز!"
log "=========================================="
echo ""
echo "اطلاعات ورود:"
echo "  Email: admin@dezkala.com"
echo "  Password: admin@12345"
echo ""
echo "URL: https://acc.dezkala.com"
echo ""
echo "لطفاً این رمز را تغییر دهید!"
echo ""
log "لاگ نصب در: $LOG_FILE"
