<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class SyncLog extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'sync_logs';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'sync_type',
        'direction',
        'records_count',
        'status',
        'error_message',
        'last_sync',
        'synced_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'records_count' => 'integer',
        'last_sync' => 'datetime',
        'synced_at' => 'datetime',
    ];

    /**
     * Sync types.
     *
     * @var array
     */
    public const TYPES = [
        'inventory' => 'موجودی',
        'price' => 'قیمت',
        'product' => 'محصول',
    ];

    /**
     * Sync directions.
     *
     * @var array
     */
    public const DIRECTIONS = [
        'send' => 'ارسال',
        'receive' => 'دریافت',
    ];

    /**
     * Sync statuses.
     *
     * @var array
     */
    public const STATUSES = [
        'success' => 'موفق',
        'failed' => 'ناموفق',
        'pending' => 'در انتظار',
    ];

    /**
     * Get type label in Persian.
     */
    public function getTypeLabel()
    {
        return self::TYPES[$this->sync_type] ?? $this->sync_type;
    }

    /**
     * Get direction label in Persian.
     */
    public function getDirectionLabel()
    {
        return self::DIRECTIONS[$this->direction] ?? $this->direction;
    }

    /**
     * Get status label in Persian.
     */
    public function getStatusLabel()
    {
        return self::STATUSES[$this->status] ?? $this->status;
    }

    /**
     * Scope: get successful syncs.
     */
    public function scopeSuccessful($query)
    {
        return $query->where('status', 'success');
    }

    /**
     * Scope: get failed syncs.
     */
    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    /**
     * Scope: get pending syncs.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope: get recent syncs.
     */
    public function scopeRecent($query, $minutes = 60)
    {
        return $query->where('synced_at', '>', now()->subMinutes($minutes))
            ->orderBy('synced_at', 'desc');
    }
}
