<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class StockCount extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'stock_counts';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'warehouse_id',
        'count_date',
        'start_time',
        'end_time',
        'status',
        'counted_by',
        'verified_by',
        'notes',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'count_date' => 'date',
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the warehouse for the stock count.
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id', 'id');
    }

    /**
     * Get the items for the stock count.
     */
    public function items()
    {
        return $this->hasMany(StockCountItem::class, 'count_id', 'id');
    }

    /**
     * Get the user who counted the stock.
     */
    public function countedBy()
    {
        return $this->belongsTo(User::class, 'counted_by', 'id');
    }

    /**
     * Get the user who verified the count.
     */
    public function verifiedBy()
    {
        return $this->belongsTo(User::class, 'verified_by', 'id');
    }

    /**
     * Calculate duration of the count in minutes.
     */
    public function getDuration()
    {
        if (!$this->start_time || !$this->end_time) return null;
        return $this->start_time->diffInMinutes($this->end_time);
    }

    /**
     * Get total variance for the count.
     */
    public function getTotalVariance()
    {
        return $this->items()->sum('variance');
    }

    /**
     * Get count accuracy percentage.
     */
    public function getAccuracy()
    {
        $items = $this->items();
        $totalItems = $items->count();
        
        if ($totalItems == 0) return 100;
        
        $accurateItems = $items->where('variance', 0)->count();
        return round(($accurateItems / $totalItems) * 100, 2);
    }

    /**
     * Scope: get in-progress counts.
     */
    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    /**
     * Scope: get completed counts.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }
}
