<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class PriceAuditLog extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'price_audit_logs';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'old_price',
        'new_price',
        'change_type',
        'changed_by',
        'change_date',
        'change_reason',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'old_price' => 'decimal:0',
        'new_price' => 'decimal:0',
        'change_date' => 'datetime',
    ];

    /**
     * Get price difference.
     */
    public function getPriceDifference()
    {
        return $this->new_price - $this->old_price;
    }

    /**
     * Get percentage change.
     */
    public function getPercentageChange()
    {
        if ($this->old_price == 0) return 0;
        return (($this->getPriceDifference()) / $this->old_price) * 100;
    }

    /**
     * Get the user who made the change.
     */
    public function changedBy()
    {
        return $this->belongsTo(User::class, 'changed_by', 'id');
    }

    /**
     * Scope: get price increases.
     */
    public function scopeIncreases($query)
    {
        return $query->whereRaw('new_price > old_price');
    }

    /**
     * Scope: get price decreases.
     */
    public function scopeDecreases($query)
    {
        return $query->whereRaw('new_price < old_price');
    }

    /**
     * Scope: get by product.
     */
    public function scopeByProduct($query, $productId)
    {
        return $query->where('product_id', $productId);
    }

    /**
     * Scope: get by date range.
     */
    public function scopeDateBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('change_date', [$startDate, $endDate]);
    }
}
