<?php

namespace App\Http\Controllers\Api;

use App\Models\ProductPrice;
use App\Models\PriceAuditLog;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class PriceController extends Controller
{
    /**
     * Get price list with filters.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        $query = ProductPrice::query();

        // Filter by product
        if ($request->has('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        // Filter by active discounts
        if ($request->has('with_discount') && $request->with_discount === 'true') {
            $query->withActiveDiscount();
        }

        $prices = $query->orderBy('created_at', 'desc')->paginate(15);

        return response()->json([
            'success' => true,
            'message' => 'قیمت‌ها دریافت‌شده',
            'data' => $prices->items(),
            'pagination' => [
                'current_page' => $prices->currentPage(),
                'total' => $prices->total(),
                'per_page' => $prices->perPage(),
                'last_page' => $prices->lastPage(),
            ],
        ]);
    }

    /**
     * Get single price details.
     *
     * @param ProductPrice $price
     * @return JsonResponse
     */
    public function show(ProductPrice $price): JsonResponse
    {
        $effectivePrice = $price->getEffectivePrice();
        $profitMargin = $price->getProfitMargin();

        return response()->json([
            'success' => true,
            'message' => 'قیمت دریافت‌شده',
            'data' => array_merge($price->toArray(), [
                'effective_price' => $effectivePrice,
                'profit_margin_percentage' => $profitMargin,
            ]),
        ]);
    }

    /**
     * Create or update product price.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'base_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'wholesale_price' => 'sometimes|numeric|min:0',
            'discount_percentage' => 'sometimes|numeric|min:0|max:100',
            'discount_start' => 'sometimes|date',
            'discount_end' => 'sometimes|date|after:discount_start',
            'currency' => 'sometimes|string|size:3',
        ]);

        // Create new price record
        $price = ProductPrice::create(array_merge($validated, [
            'effective_date' => now(),
            'currency' => $validated['currency'] ?? 'IRR',
        ]));

        // Log the change
        PriceAuditLog::create([
            'product_id' => $validated['product_id'],
            'old_price' => null,
            'new_price' => $validated['selling_price'],
            'change_type' => 'creation',
            'changed_by' => auth()->id(),
            'change_reason' => 'ایجاد قیمت جدید',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'قیمت ایجاد شد',
            'data' => $price,
        ], 201);
    }

    /**
     * Update price.
     *
     * @param Request $request
     * @param ProductPrice $price
     * @return JsonResponse
     */
    public function update(Request $request, ProductPrice $price): JsonResponse
    {
        $validated = $request->validate([
            'base_price' => 'sometimes|numeric|min:0',
            'selling_price' => 'sometimes|numeric|min:0',
            'wholesale_price' => 'sometimes|numeric|min:0',
            'discount_percentage' => 'sometimes|numeric|min:0|max:100',
            'discount_start' => 'sometimes|date',
            'discount_end' => 'sometimes|date',
        ]);

        // Store old price for audit
        $oldPrice = $price->selling_price;

        $price->update($validated);

        // Log the change if selling price changed
        if (isset($validated['selling_price']) && $validated['selling_price'] != $oldPrice) {
            PriceAuditLog::create([
                'product_id' => $price->product_id,
                'old_price' => $oldPrice,
                'new_price' => $validated['selling_price'],
                'change_type' => 'update',
                'changed_by' => auth()->id(),
                'change_reason' => $request->get('reason', 'تعدیل قیمت'),
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'قیمت بروزرسانی شد',
            'data' => $price->fresh(),
        ]);
    }

    /**
     * Get price history for product.
     *
     * @param Request $request
     * @param int $productId
     * @return JsonResponse
     */
    public function history(Request $request, int $productId): JsonResponse
    {
        $query = PriceAuditLog::where('product_id', $productId);

        // Date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('change_date', [
                $request->start_date,
                $request->end_date,
            ]);
        }

        $history = $query->orderBy('change_date', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'message' => 'تاریخچه قیمت دریافت‌شده',
            'data' => $history->items(),
            'pagination' => [
                'current_page' => $history->currentPage(),
                'total' => $history->total(),
                'per_page' => $history->perPage(),
                'last_page' => $history->lastPage(),
            ],
        ]);
    }

    /**
     * Apply discount.
     *
     * @param Request $request
     * @param ProductPrice $price
     * @return JsonResponse
     */
    public function applyDiscount(Request $request, ProductPrice $price): JsonResponse
    {
        $validated = $request->validate([
            'discount_percentage' => 'required|numeric|min:0|max:100',
            'discount_start' => 'required|date',
            'discount_end' => 'required|date|after:discount_start',
        ]);

        $price->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'تخفیف اعمال شد',
            'data' => [
                'original_price' => $price->selling_price,
                'discount_percentage' => $validated['discount_percentage'],
                'discounted_price' => $price->getEffectivePrice(),
                'savings' => $price->selling_price - $price->getEffectivePrice(),
            ],
        ]);
    }

    /**
     * Remove discount.
     *
     * @param ProductPrice $price
     * @return JsonResponse
     */
    public function removeDiscount(ProductPrice $price): JsonResponse
    {
        $price->update([
            'discount_percentage' => 0,
            'discount_start' => null,
            'discount_end' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'تخفیف حذف شد',
            'data' => $price->fresh(),
        ]);
    }

    /**
     * Get price analytics.
     *
     * @return JsonResponse
     */
    public function analytics(): JsonResponse
    {
        $totalPriceChanges = PriceAuditLog::count();
        $increases = PriceAuditLog::increases()->count();
        $decreases = PriceAuditLog::decreases()->count();
        $avgProfitMargin = ProductPrice::selectRaw('SUM(selling_price - base_price) / COUNT(*) as avg_margin')
            ->value('avg_margin') ?? 0;

        return response()->json([
            'success' => true,
            'message' => 'آمار قیمت‌گذاری',
            'data' => [
                'total_price_changes' => $totalPriceChanges,
                'price_increases' => $increases,
                'price_decreases' => $decreases,
                'average_profit_margin' => $avgProfitMargin,
            ],
        ]);
    }

    /**
     * Sync prices with main store.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function sync(Request $request): JsonResponse
    {
        try {
            // Sync logic here
            return response()->json([
                'success' => true,
                'message' => 'قیمت‌ها با موفقیت هماهنگ شد',
                'synced_records' => 0,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در هماهنگی: ' . $e->getMessage(),
            ], 500);
        }
    }
}
