<?php

namespace App\Http\Controllers\Api;

use App\Models\ProductBatch;
use App\Models\ExpirationAlert;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Carbon\Carbon;

class ExpirationController extends Controller
{
    /**
     * Get batches list.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        $query = ProductBatch::query();

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by warehouse
        if ($request->has('warehouse_id')) {
            $query->where('warehouse_id', $request->warehouse_id);
        }

        // Filter: expiring soon
        if ($request->has('expiring_soon') && $request->expiring_soon === 'true') {
            $query->expiringoon(30);
        }

        $batches = $query->orderBy('expiration_date', 'asc')->paginate(15);

        return response()->json([
            'success' => true,
            'message' => 'دسته‌های محصول دریافت‌شده',
            'data' => $batches->items(),
            'pagination' => [
                'current_page' => $batches->currentPage(),
                'total' => $batches->total(),
                'per_page' => $batches->perPage(),
                'last_page' => $batches->lastPage(),
            ],
        ]);
    }

    /**
     * Get single batch.
     *
     * @param ProductBatch $batch
     * @return JsonResponse
     */
    public function show(ProductBatch $batch): JsonResponse
    {
        return response()->json([
            'success' => true,
            'message' => 'دسته دریافت‌شده',
            'data' => array_merge($batch->toArray(), [
                'days_to_expiry' => $batch->getDaysToExpiry(),
                'is_expired' => $batch->isExpired(),
                'needs_warning' => $batch->needsWarning(),
                'is_critical' => $batch->isCritical(),
            ]),
        ]);
    }

    /**
     * Create new batch.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'product_id' => 'required|integer',
            'batch_number' => 'required|string|unique:product_batches,batch_number',
            'warehouse_id' => 'required|exists:warehouses,id',
            'quantity' => 'required|integer|min:1',
            'manufacturing_date' => 'sometimes|date',
            'expiration_date' => 'required|date|after:today',
        ]);

        $batch = ProductBatch::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'دسته ایجاد شد',
            'data' => $batch,
        ], 201);
    }

    /**
     * Update batch.
     *
     * @param Request $request
     * @param ProductBatch $batch
     * @return JsonResponse
     */
    public function update(Request $request, ProductBatch $batch): JsonResponse
    {
        $validated = $request->validate([
            'quantity' => 'sometimes|integer|min:0',
            'status' => 'sometimes|in:active,expired,recalled,disposed',
            'manufacturing_date' => 'sometimes|date',
        ]);

        $batch->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'دسته بروزرسانی شد',
            'data' => $batch->fresh(),
        ]);
    }

    /**
     * Get expired batches.
     *
     * @return JsonResponse
     */
    public function expired(): JsonResponse
    {
        $expiredBatches = ProductBatch::expired()->paginate(20);

        return response()->json([
            'success' => true,
            'message' => 'دسته‌های منقضی دریافت‌شده',
            'data' => $expiredBatches->items(),
            'count' => ProductBatch::expired()->count(),
        ]);
    }

    /**
     * Get expiring soon batches.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function expiringSoon(Request $request): JsonResponse
    {
        $days = $request->get('days', 30);
        $expiringBatches = ProductBatch::expiringoon($days)->paginate(20);

        return response()->json([
            'success' => true,
            'message' => 'دسته‌های به‌زودی منقضی دریافت‌شده',
            'data' => $expiringBatches->items(),
            'count' => ProductBatch::expiringoon($days)->count(),
            'threshold_days' => $days,
        ]);
    }

    /**
     * Get expiration alerts.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function alerts(Request $request): JsonResponse
    {
        $query = ExpirationAlert::query();

        // Filter by level
        if ($request->has('level')) {
            $query->byLevel($request->level);
        }

        // Filter unacknowledged
        if ($request->has('unacknowledged') && $request->unacknowledged === 'true') {
            $query->unacknowledged();
        }

        $alerts = $query->orderBy('alert_date', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'message' => 'هشدارهای انقضا دریافت‌شده',
            'data' => $alerts->items(),
            'summary' => [
                'total' => ExpirationAlert::count(),
                'warning' => ExpirationAlert::where('alert_level', 'warning')->count(),
                'critical' => ExpirationAlert::where('alert_level', 'critical')->count(),
                'expired' => ExpirationAlert::where('alert_level', 'expired')->count(),
                'unacknowledged' => ExpirationAlert::unacknowledged()->count(),
            ],
        ]);
    }

    /**
     * Create expiration alert.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function createAlert(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'batch_id' => 'required|exists:product_batches,id',
            'alert_level' => 'required|in:warning,critical,expired',
            'days_remaining' => 'required|integer',
            'sent_to' => 'sometimes|string',
        ]);

        $alert = ExpirationAlert::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'هشدار ایجاد شد',
            'data' => $alert,
        ], 201);
    }

    /**
     * Acknowledge alert.
     *
     * @param ExpirationAlert $alert
     * @return JsonResponse
     */
    public function acknowledgeAlert(ExpirationAlert $alert): JsonResponse
    {
        $alert->acknowledge(auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'هشدار تایید شد',
            'data' => $alert,
        ]);
    }

    /**
     * Record action for alert.
     *
     * @param Request $request
     * @param ExpirationAlert $alert
     * @return JsonResponse
     */
    public function recordAction(Request $request, ExpirationAlert $alert): JsonResponse
    {
        $validated = $request->validate([
            'action' => 'required|string',
        ]);

        $alert->recordAction($validated['action'], auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'اقدام ثبت شد',
            'data' => $alert,
        ]);
    }

    /**
     * Get expiration report.
     *
     * @return JsonResponse
     */
    public function report(): JsonResponse
    {
        $report = [
            'total_batches' => ProductBatch::active()->count(),
            'expired_batches' => ProductBatch::expired()->count(),
            'expiring_30_days' => ProductBatch::expiringoon(30)->count(),
            'expiring_7_days' => ProductBatch::expiringoon(7)->count(),
            'active_alerts' => ExpirationAlert::unacknowledged()->count(),
            'critical_alerts' => ExpirationAlert::critical()->unacknowledged()->count(),
        ];

        return response()->json([
            'success' => true,
            'message' => 'گزارش انقضا',
            'data' => $report,
        ]);
    }
}
