# راهنمای استقرار و نصب

## مرحله 1: پیش‌نیازها

### الزامات سیستم
- PHP >= 8.1
- MySQL >= 5.7
- Composer >= 2.0
- Node.js >= 16 (برای بخش‌های فرانتی)
- Redis (برای queue و caching - اختیاری)

### بررسی نسخه‌ها
```bash
php -v
mysql --version
composer --version
node -v
npm -v
```

---

## مرحله 2: دریافت پروژه

### از GitHub
```bash
cd /home/dezkalac/public_html
git clone https://github.com/dezkala/accounting-system.git acc
cd acc
```

### یا از بکاپ
```bash
cd /home/dezkalac/public_html
cp -r ACCOUNTING_SYSTEM acc
cd acc
```

---

## مرحله 3: نصب Dependencies

### نصب Composer
```bash
composer install
```

### نصب npm packages
```bash
npm install
```

### Build frontend assets
```bash
npm run production
```

---

## مرحله 4: تنظیمات محیط

### کپی کردن فایل .env
```bash
cp .env.example .env
```

### تولید Application Key
```bash
php artisan key:generate
```

### تولید JWT Secret
```bash
php artisan jwt:secret
```

### ویرایش فایل .env

```env
# App Configuration
APP_NAME="سیستم حسابداری دزکالا"
APP_ENV=production
APP_KEY=base64:YOUR_KEY_HERE
APP_DEBUG=false
APP_URL=https://acc.dezkala.com

# Database Configuration - Main Store
DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=dezkalac_seo
DB_USERNAME=dezkalac
DB_PASSWORD=YOUR_PASSWORD

# Database Configuration - Accounting System
ACCOUNTING_DB_CONNECTION=mysql
ACCOUNTING_DB_HOST=localhost
ACCOUNTING_DB_PORT=3306
ACCOUNTING_DB_DATABASE=dezkalac_accounting
ACCOUNTING_DB_USERNAME=dezkalac
ACCOUNTING_DB_PASSWORD=YOUR_PASSWORD

# Cache
CACHE_DRIVER=redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

# Queue
QUEUE_CONNECTION=redis

# Mail Configuration
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io
MAIL_PORT=587
MAIL_USERNAME=YOUR_USERNAME
MAIL_PASSWORD=YOUR_PASSWORD
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS="accounting@dezkala.com"
MAIL_FROM_NAME="سیستم حسابداری"

# JWT Configuration
JWT_SECRET=YOUR_JWT_SECRET
JWT_ALGORITHM=HS256
JWT_EXPIRES_IN=3600

# Sync Configuration
SYNC_ENABLED=true
SYNC_INTERVAL=300
SYNC_STORE_URL=https://dezkala.com
SYNC_STORE_API_KEY=YOUR_API_KEY

# Feature Flags
FEATURE_TWO_FACTOR_AUTH=true
FEATURE_BATCH_IMPORT=true
FEATURE_ADVANCED_REPORTS=true
FEATURE_REAL_TIME_SYNC=true
```

---

## مرحله 5: تنظیم دیتابیس

### ایجاد دیتابیس
```bash
mysql -u dezkalac -p -e "CREATE DATABASE dezkalac_accounting CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
```

### اجرای Migrations
```bash
php artisan migrate --database=mysql
```

### Seeding دیتابیس (اختیاری)
```bash
php artisan db:seed
```

### اجرای فایل SQL (اگر موجود): 
```bash
mysql -u dezkalac -p dezkalac_accounting < database_schema.sql
```

---

## مرحله 6: تنظیمات وب‌سرور

### برای Apache/cPanel

**1. بروزرسانی DocumentRoot:**
```bash
# edit acc folder permissions
chmod -R 755 /home/dezkalac/public_html/acc
chmod -R 777 /home/dezkalac/public_html/acc/storage
chmod -R 777 /home/dezkalac/public_html/acc/bootstrap/cache
```

**2. ایجاد فایل .htaccess:**

```apache
<IfModule mod_rewrite.c>
  <IfModule mod_negotiation.c>
    Options -MultiViews -Indexes
  </IfModule>

  RewriteEngine On

  # Handle Authorization Header
  RewriteCond %{HTTP:Authorization} .
  RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

  # Redirect Trailing Slashes If Not A Folder...
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteCond %{REQUEST_URI} (.+)/$
  RewriteRule ^ %1 [L,R=301]

  # Send Requests To Front Controller...
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteRule ^ index.php [L]
</IfModule>
```

**3. تنظیم Public DocumentRoot در cPanel:**
- برو به Addon Domains
- ایجاد زیر‌دامنه `acc.dezkala.com`
- DocumentRoot را بر روی `/public_html/acc/public` تنظیم کن

### برای Nginx

```nginx
server {
    listen 80;
    listen [::]:80;
    server_name acc.dezkala.com;
    
    root /home/dezkalac/public_html/acc/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/run/php-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    # SSL (بعد از تنظیم)
    # listen 443 ssl http2;
    # ssl_certificate /path/to/cert;
    # ssl_certificate_key /path/to/key;
}
```

---

## مرحله 7: SSL/HTTPS

### استفاده از Let's Encrypt از طریق cPanel

```bash
# یا از cPanel AutoSSL استفاده کن
```

### یا دستی:
```bash
# نصب Certbot
sudo apt-get install certbot python3-certbot-apache

# صدور گواهی
sudo certbot certonly -a webroot -w /home/dezkalac/public_html/acc/public -d acc.dezkala.com
```

---

## مرحله 8: تنظیمات Cron Jobs

### بروز رسانی توسط supervisor

```bash
# نصب Supervisor
sudo apt-get install supervisor

# ایجاد فایل تنظیمات
sudo nano /etc/supervisor/conf.d/accounting-queue.conf
```

**محتوای فایل:**
```ini
[program:accounting-queue]
process_name=%(program_name)s_%(process_num)02d
command=php /home/dezkalac/public_html/acc/artisan queue:work redis --sleep=3 --tries=3
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
numprocs=4
redirect_stderr=true
stdout_logfile=/home/dezkalac/logs/queue.log
stopwaitsecs=3600
user=dezkalac
```

### یا استفاده از Cron

```bash
# ویرایش crontab
crontab -e

# اضافه کردن این خطوط:
* * * * * cd /home/dezkalac/public_html/acc && php artisan schedule:run >> /dev/null 2>&1
*/5 * * * * cd /home/dezkalac/public_html/acc && php artisan sync:inventory >> /dev/null 2>&1
*/10 * * * * cd /home/dezkalac/public_html/acc && php artisan batches:check-expiration >> /dev/null 2>&1
0 2 * * * cd /home/dezkalac/public_html/acc && php artisan backups:clean >> /dev/null 2>&1
```

---

## مرحله 9: سرویس‌های اضافی

### Redis (برای Caching و Queue)

```bash
# نصب
sudo apt-get install redis-server

# شروع سرویس
sudo systemctl start redis-server
sudo systemctl enable redis-server

# بررسی وضعیت
redis-cli ping
```

### مدیریت لاگ‌ها

```bash
# ایجاد directory برای لاگ‌ها
mkdir -p /home/dezkalac/logs/accounting
chmod 777 /home/dezkalac/logs/accounting

# تنظیم log rotation
cat > /etc/logrotate.d/accounting << EOF
/home/dezkalac/logs/accounting/*.log {
    daily
    missingok
    rotate 14
    compress
    delaycompress
    notifempty
    create 0640 dezkalac dezkalac
}
EOF
```

---

## مرحله 10: بررسی نهایی و تست

### Test API Health Check
```bash
curl https://acc.dezkala.com/api/health
```

### چک کردن وضعیت Migrations
```bash
php artisan migrate:status
```

### چک کردن وضعیت Cache
```bash
php artisan cache:clear
php artisan config:cache
php artisan view:cache
```

### Create Initial Admin User
```bash
php artisan tinker

# سپس در tinker:
$user = new App\Models\User();
$user->name = 'Admin';
$user->email = 'admin@dezkala.com';
$user->phone = '09166809066';
$user->role = 'admin';
$user->password = Hash::make('admin@12345');
$user->warehouse_id = 1;
$user->status = 'active';
$user->save();

exit
```

---

## مرحله 11: مانیتورینگ و نگهداری

### بکاپ دیتابیس

```bash
# ایجاد script backup
cat > /home/dezkalac/backup-accounting.sh << 'EOF'
#!/bin/bash
DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/home/dezkalac/backups/accounting"
mkdir -p $BACKUP_DIR

mysqldump -u dezkalac -p$DB_PASSWORD dezkalac_accounting | gzip > $BACKUP_DIR/accounting_$DATE.sql.gz

# حفظ فقط بکاپ‌های 30 روز اخیر
find $BACKUP_DIR -type f -mtime +30 -delete
EOF

chmod +x /home/dezkalac/backup-accounting.sh

# اجرای روزانه
echo "0 3 * * * /home/dezkalac/backup-accounting.sh >> /home/dezkalac/logs/backup.log 2>&1" | crontab -
```

### مانیتورینگ Performance

```bash
# نصب Laravel Debugbar (برای development)
composer require barryvdh/laravel-debugbar --dev

# نصب APM (برای production)
composer require elastic/apm-agent
```

---

## حل مشکلات رایج

### 500 Error
```bash
# بررسی log
tail -f storage/logs/laravel.log

# چک کردن permissions
chmod -R 777 storage bootstrap/cache
```

### Database Connection Error
```bash
# بررسی اتصال
php artisan tinker
>>> DB::connection()->getPdo()

# بررسی connection string در .env
```

### Permission Denied
```bash
# تصحیح مالکیت فایل‌ها
chown -R dezkalac:dezkalac /home/dezkalac/public_html/acc
```

### Queue Not Working
```bash
# تست queue
php artisan queue:work --verbose

# بررسی Redis
redis-cli
```

---

## نکات امنیتی

1. ✅ تنظیم `APP_DEBUG=false` در production
2. ✅ استفاده از HTTPS/SSL
3. ✅ تنظیم firewall برای محدود کردن IP
4. ✅ تنظیم rate limiting برای API
5. ✅ نگهداری منظم بکاپ‌های دیتابیس
6. ✅ به روزرسانی منظم dependencies
7. ✅ تفعیل Two-Factor Authentication
8. ✅ محدود کردن CORS

```php
// در config/cors.php
'allowed_origins' => ['https://dezkala.com', 'https://acc.dezkala.com'],
'allowed_origins_patterns' => [],
```

---

## مراجع مفید

- [Laravel Documentation](https://laravel.com/docs)
- [MySQL Documentation](https://dev.mysql.com/doc/)
- [JWT Authentication](https://jwt.io/)
- [cPanel Documentation](https://documentation.cpanel.net/)

---

**نسخه:** 1.0  
**آخرین بروزرسانی:** ۱۴۰۵/۱۰/۲۳
